<?	// copyright (C) 2006 - Matt Newberry, matt@mattnewberry.net
	// ALL RIGHTS RESERVED, except as site-licensed under contract
		
	/*  A News object is a container for Article objects; each Article is a container for paragraphs of text,
		the first of which may have it's first several words, its 'headline', singled out for special emphasis. 
		The Articles are sorted in reverse order by date, but the order may be reversed by the administrator.
		
		Public Interface:
		
				//-- constructor ---------------
			
			function News($filename);
				- creates a News object from an XML datafile; $filename should be a full path 
					relative to the executing script directory
			
				//-- display functions ---------------
			
			function printPageTitle();
				- prints an identifying string, html-encoded, for an instance of this class (from XmlSimpleBaseClass)
			
			function sortAscending();
				- sorts latest submissions last
			
			function sortDescending();
				- sorts latest submissions first
			
			function getNextArticle();
			
			function Article.printID();
			
			function Article.printDate();
				- Month DD, YYYY format
			
			function Article.printTimestamp();
				- Month DD, YYYY HH:MM:SS format
			
			function Article.printHeadline();
			
			function Article.printParagraphs();
			
			function Article.printHeadlineTruncated($maxlen);
			
			function Article.moreParagraphs();
			
			function Article.printNextParagraph();
			
				//-- error handling ---------------
			
			function parseFailed();
				- returns false if the News object was successfully parsed from xml, otherwise true (from XmlSimpleBaseClass)
			
			function getParseError();
				- returns error code from the PHP xml parser; 0 if no error, 8001 if input file not found 
					or not readable (from XmlSimpleBaseClass)
			
			function printParseMessage();
				- prints error message from the PHP xml parser, html-encoded (from XmlSimpleBaseClass)
			
				//-- admin functions ---------------
			
			function addArticle($headline, $body);
			
			function updateArticle($articleid, $headline, $body);
			
			function getArticle($articleid);
			
			function deleteArticle($articleid);
			
			function Article.setTimestamp($timestamp);
			
			function Article.populate($headline, $body);
			
			function Article.addParagraphs($text);
			
			function save();
				- save data back to disk under original filename (from XmlSimpleBaseClass)
		
		
		Typical usage in HTML display context:
		
			$picture = new News("data/picture.xml");
			while (($article = $picture->getNextArticle()) != null) {
				$article.printHeadline();
				while ($article->moreParagraphs())
					$article->printNextParagraph();
			}
		
		
		XML Data Format:
		
			<news>
				<pagetitle>TEXT</pagetitle>
				<articles>
					<article>
						<timestamp>NUMBER</timestamp>
						<headline>TEXT</headline>
						<paragraphs>
							<paragraph>TEXT</paragraph>
										.
										.
						</paragraphs>
					</articles>
						.
						.
				</article>
			</news>
	*/
	require_once("XmlSimpleBaseClass.class.php");
	
	class News extends XmlSimpleBaseClass {
		var $xml_articles = array();		// associative; key = submission timestamp
		var $_article;
		
		function News($filename) {
			$this->XmlSimpleBaseClass($filename);
			$this->sortDescending();
		}
		
		function _p_start_element($parser, $element, &$attributes) {
		 	parent::_p_start_element($parser, $element, &$attributes);
			switch ($element) {
				case "article":
					$this->_article = new Article();
					break;
				case "paragraph":
					$this->_article->xml_paragraphs[] = "";
					++$this->_article->_index;
			}
		}
		
		function _p_cdata($parser, $text) {
			switch ($this->_tag) {
				case "timestamp":
					$this->_article->setTimestamp($text);
					break;
				case "headline":
					$this->_article->xml_headline .= $text;
					break;
				case "paragraph":
					$this->_article->xml_paragraphs[$this->_article->_index] .= $text;
					break;
				case "pagetitle":
					$this->xml_pagetitle .= $text;
					break;
			}
		}
		
				// decode entities as character data
		function _p_default($parser, $ent) {
			$this->_p_cdata($parser, html_entity_decode($ent));
		}
		
		function _p_end_element($parser, $element) {
		 	parent::_p_end_element($parser, $element);
			switch ($element) {
				case "article":
					$this->xml_articles[$this->_article->xml_timestamp] = $this->_article;
					break;
				case "articles":
					krsort($this->xml_articles);
					break;
			}
		}
		
		function getNextArticle() {
			if (list($key, $article) = each($this->xml_articles)) {
				return $article;
			}
			else {
				reset($this->xml_articles);
				return false;
			}
		}
		
		function addArticle($headline, $body) {
			$article = new Article();
			$stamp = time();
			$article->setTimestamp($stamp);
			$article->populate($headline, $body);
			$this->xml_articles[$stamp] = $article;
			krsort($this->xml_articles);
			$this->save();
		}
		
		function updateArticle($articleid, $headline, $body) {
			$article =& $this->getArticle($articleid);	// byref
			$article->populate($headline, $body);
			$this->save();
		}
		
		function getArticle($articleid) {
			return $this->xml_articles[$articleid];
		}
		
		function deleteArticle($articleid) {
			foreach ($this->xml_articles as $a) {
				if ($a->xml_timestamp == $articleid)
					unset($this->xml_articles[$articleid]);
			}
			krsort($this->xml_articles);
			$this->save();
		}
		
		function sortAscending() {
			ksort($this->xml_articles);
		}
		
		function sortDescending() {
			krsort($this->xml_articles);
		}
	}
	
		// ----- Article class -----
	
	class Article {
		var $xml_timestamp = "";
		var $xml_headline = "";
		var $xml_paragraphs = array();		// indexed
		var $_index = -1;
		
		function Article() {
			;
		}
		
		function setTimestamp($timestamp) {
			$this->xml_timestamp = $timestamp;
		}
		
		function populate($headline, $body) {
			$this->xml_headline = $headline;
			$this->addParagraphs($body);
		}
		
		function addParagraphs($text) {
			$this->xml_paragraphs = array();
			while (($index = strpos($text, "\r\n\r\n")) > 0) {
				$this->xml_paragraphs[] = trim(substr($text, 0, $index));
				$text = trim(substr($text, $index+2));
			}
			$this->xml_paragraphs[] = $text;
		}
		
		function printID() {
			print htmlspecialchars(trim($this->xml_timestamp));
		}
		
		function printDate() {
			$date = getdate($this->xml_timestamp);
			$datestring = $date["month"]." ".$date["mday"].", ".$date["year"];
			print htmlspecialchars($datestring);
		}
		
		function printTimestamp() {
			$date = getdate($this->xml_timestamp);
			$datestring = $date["month"]." ".$date["mday"].", ".$date["year"]." ".$date["hours"].":".$date["minutes"].":".$date["seconds"];
			print htmlspecialchars($datestring);
		}
		
		function printHeadline() {
			print htmlspecialchars(trim($this->xml_headline));
		}
		
		function printParagraphs() {
			$text = "";
			foreach ($this->xml_paragraphs as $p)
				$text .= trim($p)."\r\n\r\n";
			print $this->htmlFilterSelective($text);
		}
		
		function printParagraphsForEdit() {
			$text = "";
			foreach ($this->xml_paragraphs as $p)
				$text .= trim($p)."\r\n\r\n";
			print htmlspecialchars($text);
		}
		
		function printHeadlineTruncated($maxlen) {
			print htmlspecialchars(substr($this->xml_headline, 0, $maxlen));
			if (strlen($this->xml_headline) > $maxlen)
				print "...";
		}
		
		function moreParagraphs() {
			if (current($this->xml_paragraphs))
				return true;
			else {
				reset($this->xml_paragraphs);
				return false;
			}
		}
		
		function printNextParagraph() {
			print $this->htmlFilterSelective(trim(current($this->xml_paragraphs)));
			next($this->xml_paragraphs);
		}
		
		function printNextParagraphForEdit() {
			print htmlspecialchars(trim(current($this->xml_paragraphs)));
			next($this->xml_paragraphs);
		}
	}
?>